using System;
using System.Xml;

using gov.va.med.vbecs.DAL.VistALink.OpenLibrary;

namespace gov.va.med.vbecs.DAL.VistALink.OpenLibrary.Messages
{
	/// <summary>
	/// This is a base class for all VistALink response messages.
	/// </summary>
	public abstract class BaseResponseMessage : VistALinkMessage
	{
		/// <summary>
		/// Message response type.
		/// </summary>
		private readonly string _responseType;

		/// <summary>
		/// VistALink message response status.
		/// </summary>
		protected ResponseStatus _responseStatus; 

		// Constants used in XML message serialization/deserialization
		private const string XMLCONSTS_RESPONSE_NODE_NAME = "Response";
		private const string XMLCONSTS_TYPE_RESPONSE_ATTRIBUTE_NAME = "type";
		private const string XMLCONSTS_STATUS_RESPONSE_ATTRIBUTE_NAME = "status";

		/// <summary>
		/// Contstructor specifying minimal required VistALink response message parameters. 
		/// </summary>
		/// <param name="messageType">VistALink message type</param>
		/// <param name="sourceXSD">VistALink message XSD name</param>
		/// <param name="responseType">Response type</param>
		/// <param name="responseStatus">Response status. Normally will indicate status of preceding request. </param>
		protected BaseResponseMessage( string messageType, string sourceXSD, string responseType, ResponseStatus responseStatus ) 
			: base( messageType, sourceXSD )
		{
			if( responseType == null )
				throw( new ArgumentNullException( "responseType" ) );

			if( responseStatus == null )
				throw( new ArgumentNullException( "responseStatus" ) );

			_responseType = responseType;
			_responseStatus = responseStatus;
		}

		/// <summary>
		/// Constructor extending standard VistALink message deserialization constructor. 
		/// Allows recreating message object from XML document.
		/// </summary>
		/// <param name="messageType">VistALink message type</param>
		/// <param name="sourceXSD">VistALink message XSD name</param>
		/// <param name="responseType">VistALink response type</param>
		/// <param name="sourceDoc">Source XML document</param>
		protected BaseResponseMessage( string messageType, string sourceXSD, string responseType, XmlDocument sourceDoc ) 
			: base( messageType, sourceXSD, sourceDoc )
		{
			if( responseType == null )
				throw( new ArgumentNullException( "responseType" ) );

			XmlElement _responseElement = XmlParseGetResponseElement( sourceDoc );

			XmlUtility.ParseCheckRequiredAttributeValue( _responseElement, XMLCONSTS_TYPE_RESPONSE_ATTRIBUTE_NAME, responseType );
			_responseType = responseType;

			_responseStatus = ResponseStatus.Parse( XmlUtility.ParseGetRequiredAttributeValue( _responseElement, XMLCONSTS_STATUS_RESPONSE_ATTRIBUTE_NAME ) );
		}

		/// <summary>
		/// XML parsing utility method. Returns reference to response element in 
		/// the specified XML document containing XML representation of VistALink 
		/// base response message.
		/// 
		/// If element not found or more than one instance was found - 
		/// XmlParseException exception will be thrown.
		/// </summary>
		/// <param name="sourceDoc">Source XML document</param>
		/// <returns>Reference to response element</returns>
		protected static XmlElement XmlParseGetResponseElement( XmlDocument sourceDoc )
		{
			return XmlUtility.ParseGetRequiredElementByUniqueTagName( sourceDoc, XMLCONSTS_RESPONSE_NODE_NAME );
		}

		/// <summary>
		/// XML parsing method used during message recognition and deserialization.
		/// Returns message response type string contained in supplied XML document.
		/// Source XML document must have a response element, otherwise method will throw parsing exception.
		/// </summary>
		/// <param name="sourceDoc"></param>
		/// <returns></returns>
		public static string XmlParseGetResponseTypeString( XmlDocument sourceDoc )
		{
			if( sourceDoc == null )
				throw( new ArgumentNullException( "sourceDoc" ) );

			return XmlParseGetResponseElement( sourceDoc ).GetAttribute( XMLCONSTS_TYPE_RESPONSE_ATTRIBUTE_NAME );
		}

		/// <summary>
		/// XML parsing method used during message recognition and deserialization.
		/// Returns response status contained in message XML.
		/// Source XML document must have a response element, otherwise method will throw parsing exception.
		/// </summary>
		/// <param name="sourceDoc">Source XmlDocument to analyze.</param>
		/// <returns>Parsed response status contained in supplied XML.</returns>
		public static ResponseStatus XmlParseGetResponseStatus( XmlDocument sourceDoc )
		{
			if( sourceDoc == null )
				throw( new ArgumentNullException( "sourceDoc" ) );

			return ResponseStatus.Parse( XmlParseGetResponseElement( sourceDoc ).GetAttribute( XMLCONSTS_STATUS_RESPONSE_ATTRIBUTE_NAME ) );
		}

		/// <summary>
		/// Standard VistALink message XML serialization method, 
		/// translating message object into XML representation.
		/// </summary>
		/// <param name="writer">XmlWriter object to use</param>
		protected override void WriteXmlMessageBody( XmlWriter writer )
		{
			if( writer == null )
				throw( new ArgumentNullException( "writer" ) );

			writer.WriteStartElement( XMLCONSTS_RESPONSE_NODE_NAME );
			writer.WriteAttributeString( XMLCONSTS_TYPE_RESPONSE_ATTRIBUTE_NAME, _responseType );
			writer.WriteAttributeString( XMLCONSTS_STATUS_RESPONSE_ATTRIBUTE_NAME, _responseStatus.ToString() );

			WriteXmlResponseAttributes( writer );

			WriteXmlResponseNodeContent( writer );

			writer.WriteEndElement();
		}

		/// <summary>
		/// Derived classes should override this method to add attributes to response element 
		/// of XML message representation during XML serialization. 
		/// </summary>
		/// <param name="writer">XmlWriter to use</param>
		protected virtual void WriteXmlResponseAttributes( XmlWriter writer ){}

		/// <summary>
		/// Derived classes should override this method to add content to response element 
		/// of XML message representation during XML serialization. 
		/// </summary>
		/// <param name="writer">XmlWriter to use</param>
		protected virtual void WriteXmlResponseNodeContent( XmlWriter writer ){}

		/// <summary>
		/// Response status. Normally will indicate status of preceding request. 
		/// </summary>
		public ResponseStatus ResponseStatus
		{
			get
			{
				return _responseStatus;
			}
		}
	}
}
